#!/bin/bash

# Chemin du script de mise à jour
SCRIPT_PATH="/usr/local/bin/update-auto.sh"
# Construire le nom du fichier log
LOGFILE="/var/log/update-auto.log"

# Création du script de mise à jour avec commentaires et séparation dans le log
sudo tee $SCRIPT_PATH > /dev/null << 'EOF'
#!/bin/bash

# Construire le nom du fichier log
LOGFILE="/var/log/update-auto.log"

# Ajoute une ligne de séparation avec la date et l'heure dans le log
echo "=================================  DÉBUT MAJ : $(date '+%Y-%m-%d %H:%M:%S') ================================= " >> "$LOGFILE"

# Met à jour la liste des paquets disponibles
apt update >> "$LOGFILE" 2>&1

# Met à niveau tous les paquets installés vers la dernière version disponible
apt full-upgrade -y >> "$LOGFILE" 2>&1

# Tente de réparer les paquets cassés ou les dépendances manquantes
apt-get install -fy >> "$LOGFILE" 2>&1

# Nettoie les fichiers de paquets téléchargés qui ne sont plus nécessaires
apt-get autoclean >> "$LOGFILE" 2>&1

# Supprime les paquets et dépendances devenus inutiles
apt-get autoremove --purge -y >> "$LOGFILE" 2>&1

# Tente de configurer les paquets qui ne l'ont pas été correctement
dpkg --configure -a >> "$LOGFILE" 2>&1

echo "=================================  FIN   MAJ : $(date '+%Y-%m-%d %H:%M:%S') ================================= " >> "$LOGFILE"
echo >> "$LOGFILE"

# Fin du script de mise à jour
EOF

# Rend le script exécutable
sudo chmod +x $SCRIPT_PATH

# Propose plusieurs exemples de fréquences cron à l'utilisateur
echo "Exemples de fréquences au format cron :"
echo "  0 2 * * *      -> Tous les jours à 2h du matin"
echo "  0 4 * * 0      -> Chaque dimanche à 4h du matin"
echo "  30 1 * * 1-5   -> Du lundi au vendredi à 1h30 du matin"
echo "  0 */6 * * *    -> Toutes les 6 heures"
echo "  */15 * * * *   -> Toutes les 15 minutes (déconseillé pour les mises à jour système)"
echo
echo "Entrez la fréquence souhaitée (par exemple : 0 2 * * * ):"
read CRON_FREQ

# Vérifie que la fréquence n'est pas vide
if [ -z "$CRON_FREQ" ]; then
    echo "Fréquence non renseignée. Abandon."
    exit 1
fi

# Manipulation de la crontab root :
TMP_CRON=$(mktemp)
sudo crontab -l 2>/dev/null | grep -v "$SCRIPT_PATH" > "$TMP_CRON"
echo "$CRON_FREQ $SCRIPT_PATH" >> "$TMP_CRON"
sudo crontab "$TMP_CRON"
rm "$TMP_CRON"

# Ajout de la configuration logrotate
sudo tee /etc/logrotate.d/update-auto > /dev/null << 'EOF'
/var/log/update-auto.log {
    monthly
    rotate 12
    size 100k
    missingok
    notifempty
    compress
    delaycompress
    create 640 root adm
    postrotate
        # Rien à faire après la rotation
    endscript
}
EOF

echo
echo "Installation terminée !"
echo "Le script de mise à jour automatique  $SCRIPT_PATH  sera exécuté par root selon la fréquence : $CRON_FREQ"
echo "Les journaux seront enregistrés dans  $LOGFILE"
echo "La rotation des logs est gérée automatiquement chaque mois ou dès 1000 lignes environ."

