#!/usr/bin/ksh
#
#  sysinfo - display system stats (CPU speed, RAM size, etc..)
#            in a compact form
#
#  By:  Ian P. Springer, HP Software Engineer 
#       <ian_springer@am.exch.hp.com>
#       Florham Park, NJ, US
#
#  Ident: @(#) $Revision: 1.4.6 $
#

sysinfo_version=1.4.6

#  Revision History:
#
#  Date      Rev.   Who  Changes
# -----------------------------------------------------------------------
#  03/18/98  1.0.1  mm   updated to include support for HP-UX 9.x & 11.00
#  06/01/00  1.1.0  ips  revised/improved (ips)
#  09/28/00  1.1.5  ips  added better handling of large amounts of RAM;
#                        added check for adb failing to start;
#                        added SCCS ident string (ips)
#  10/25/00  1.2.0  ips  added disk & patch info and CPU model & rev.;
#                        modified to get as much info as possible if
#                        not invoked by superuser; changed the output
#                        format to make output easier to parse; added
#                        command line options (see below); added support
#                        for VirtualVault (HP-UX 10.24 & 11.04); added
#                        check for Java; numerous minor fixes &
#                        improvements
#  12/15/00  1.2.1  ips  added support for Java 1.3
#  02/02/01  1.3.0  ips  added limited support for Solaris 7 & 8;
#                        dropped support for HP-UX 9.x;
#                        script now uses ksh, since /usr/bin/sh is not
#                        the POSIX shell in Solaris 7;
#                        fixed minor bug (reported by Luis Caamano):
#                        for redundant disk discovery, use 'vgdisplay -v'
#                        instead of 'vgdisplay -v /dev/vg*', because
#                        VG names don't necessarily begin w/ 'vg'
#  06/05/01  1.4.0  ewm  added output for IP and MAC addresses
#  09/14/01  1.4.2  ips  added support for Java 1.4
#  10/04/01  1.4.3  ips  updated embedded sched.models; fixed small bug
#                        in Java detection
#  11/14/01  1.4.4  ips  RAM size detection now works for non-superuser
#  11/15/01  1.4.5  ips  improved mem.c
#  11/15/01  1.4.6  ips  added 11i patch bundle detection to 
#                        print_patches()
# -----------------------------------------------------------------------
#
#  Command-line Options:
#
#    -g : general stats (default)
#    -d : disk info
#    -n : network info
#    -p : patches installed
#    -a : all of the above
#    -q : quiet mode (no progress status; use if redirecting output to a
#         file)
#    -v : sysinfo version
#    -? : describe options
#

##############################{ FUNCTIONS }##############################

convert_bytes()
{

   if [ $# -ne 1 ] || [ -z "$1" ]; then
      return 1
   fi
   size=$1

   sizeint=`echo $size |  cut -d. -f1`
   if expr $sizeint \>= 1024 >/dev/null; then
      # convert to kilobytes
      size=`echo "scale=1; $size / 1024" | bc`
      sizeint=`echo $size | cut -d. -f1`
      if expr $sizeint \>= 1024 >/dev/null; then
         convert_kilobytes $size
         return $?
      else
         units=KB
      fi
   else
      units=bytes
   fi
   size=`echo $size | sed 's/\.0//'`
   echo $size $units
   return 0

}  # end convert_bytes()


convert_kilobytes()
{

   if [ $# -ne 1 ] || [ -z "$1" ]; then
      return 1
   fi
   size=$1

   sizeint=`echo $size |  cut -d. -f1`
   if expr $sizeint \>= 1024 >/dev/null; then
      # convert to megabytes
      size=`echo "scale=1; $size / 1024" | bc`
      sizeint=`echo $size | cut -d. -f1`
      if expr $sizeint \>= 1024 >/dev/null; then
         convert_megabytes $size
         return $?
      else
         units=MB
      fi
   else
      units=KB
   fi
   size=`echo $size | sed 's/\.0//'`
   echo $size $units
   return 0

}  # end convert_kilobytes()


convert_megabytes()
{

   if [ $# -ne 1 ] || [ -z "$1" ]; then
      return 1
   fi
   size=$1
   sizeint=`echo $size | cut -d. -f1`
   if expr $sizeint \>= 1024 >/dev/null; then
      size=`echo "scale=1; $size / 1024" | bc`
      sizeint=`echo $size | cut -d. -f1`
      if expr $sizeint \>= 1024 >/dev/null; then
         size=`echo "scale=1; $size / 1024" | bc`
	     sizeint=`echo $size | cut -d. -f1`
         units=TB
      else
         units=GB
      fi
   else
      units=MB
   fi
   size=`echo $size | sed 's/\.0//'`
   echo $size $units
   return 0

}  # end convert_megabytes()


alias setvar='read -p title ; read -p resp ; eval $(echo $resp | \
 sed "s/: */=/")'


get_kernel_stats()
{

[ "$OSNAME" = SunOS ] && return 1

MEMDEV=/dev/mem
#  if /dev/mem is not readable, return w/ error
[ ! -r $MEMDEV ] && return 1

if [ "$KERNEL_BITS" -eq 64 ]; then
   ADB="adb64 -k"
else
   ADB="adb -k"
fi

#  start adb
$ADB $KERNEL $MEMDEV |&

#  waste a few microseconds to give adb a chance to potentially fail
for foo in /etc/*; do true; false; done

#  now call jobs to clear out any completed jobs
jobs >/dev/null

#  now call jobs again to see if adb is running
if [ -z "`jobs`" ]; then
   #  try without the -k option before giving up; adb is known to fail
   #  on some K-class machines, most frequently on K460s and K570s
   if [ "$KERNEL_BITS" -eq 64 ]; then
      ADB=adb64
   else
      ADB=adb
   fi
   $ADB $KERNEL $MEMDEV |&
   #  waste a few microseconds to give adb a chance to potentially fail
   for i in /etc/*; do true; false; done
   #  now call jobs to clear out any completed jobs
   jobs >/dev/null
   if [ -z "`jobs`" ]; then
      echo "WARNING: adb failed to start;" \
           "try increasing the maxdsiz kernel param.\n" >&2
      echo
      return 2
   fi
fi

#  write out all defined kernel vars to a file
defined_kernel_vars=`mktemp`
echo "\$>$defined_kernel_vars\n \$e\n \$q" | $ADB $KERNEL $MEMDEV 2>/dev/null

#  get kernel data from adb external variables
print -p "hostname/s"            ; setvar
print -p "pdc_model_string/s"    ; setvar
print -p "cpu_revision_number/X" ; setvar
vars="cpu_arch_is_2_0 cpu_arch_is_1_1 cpu_arch_is_1_0"
vars="$vars cpu_is_a_PCXS cpu_is_a_PCXT cpu_is_a_PCXT_PRIME"
vars="$vars cpu_is_a_PCXL cpu_is_a_PCXL2 cpu_is_a_PCXU"
vars="$vars cpu_is_a_PCXU_PLUS cpu_is_a_PCXW cpu_is_a_PCXW_PLUS"
vars="$vars cpu_is_a_PIRANHA cpu_is_a_MAKO"
vars="$vars itick_per_usec processor_count"
for var in $vars; do
   if grep -q "^$var" $defined_kernel_vars; then
      print -p "$var/D"          ; setvar
   fi
done

if [ "$OSVER" -ge 1100 ]; then
   print -p phys_mem_pages/D     ; setvar
   RAM=$phys_mem_pages
else
   print -p physmem/D            ; setvar
   RAM=$physmem
fi

#  quit adb
print -p '$q'

model=$pdc_model_string
model_desc=`model -D | cut -d' ' -f2-9 | sed 's/(no description)//`
if [ -n "$model_desc" ]; then
   model="$model $model_desc"
else
   model="$model $desc"
fi

CPU_speed=$itick_per_usec
CPU_count=$processor_count

#  determine page size
eval $(grep "define *PAGESIZE" /usr/include/limits.h |
   sed 's/# *define *//;s/ /=/;s/ *//g')
if [ -z "$PAGESIZE" ]; then
   echo "PAGESIZE not explicitly defined; assuming 4 MB"
   PAGESIZE=4096
fi

if   [ "$cpu_arch_is_1_0" -eq 1 ]; then
   CPU_arch="PA-RISC 1.0"
elif [ "$cpu_arch_is_1_1" -eq 1 ]; then
   CPU_arch="PA-RISC 1.1"
elif [ "$cpu_arch_is_2_0" -eq 1 ]; then
   CPU_arch="PA-RISC 2.0"
else
   CPU_arch="unknown_architecture"
fi

# note: string comparison used below because cpu_revision_number is a
#       hex value
if [ "$cpu_arch_is_1_1" -eq 1 ] && [ "$cpu_revision_number" != 0 ]; then
   CPU_rev=`echo $cpu_revision_number | sed 's/0x//' | \
    tr "[A-Z]" "[a-z]"`
else
   CPU_rev=
fi

if   [ "$cpu_is_a_PCXS" -eq 1 ]; then
   CPU_model=PA7000
elif [ "$cpu_is_a_PCXT" -eq 1 ]; then
   CPU_model=PA7100
elif [ "$cpu_is_a_PCXT_PRIME" -eq 1 ]; then
   CPU_model=PA7200
elif [ "$cpu_is_a_PCXL" -eq 1 ]; then
   CPU_model=PA7100LC
elif [ "$cpu_is_a_PCXL2" -eq 1 ]; then
   CPU_model=PA7300LC
elif [ "$cpu_is_a_PCXU" -eq 1 ]; then
   CPU_model=PA8000
elif [ "$cpu_is_a_PCXU_PLUS" -eq 1 ]; then
   CPU_model=PA8200
elif [ "$cpu_is_a_PCXW" -eq 1 ]; then
   CPU_model=PA8500
elif [ "$cpu_is_a_PCXW_PLUS" -eq 1 ]; then
   CPU_model=PA8600
elif [ "$cpu_is_a_PIRANHA" -eq 1 ]; then
   CPU_model=PA8700
elif [ "$cpu_is_a_MAKO" -eq 1 ]; then
   CPU_model=PA8800
else
   CPU_model="unknown_model"
fi

CPU="($CPU_count) $CPU_model $CPU_speed MHz ($CPU_arch$CPU_rev)"

#  convert RAM size from pages to bytes
RAM=`echo "scale=1; $RAM * $PAGESIZE" | bc`

#  convert from bytes to a more sensible unit
RAM=`convert_bytes $RAM`

return 0

}  # end get_kernel_stats()


#
#  get_disk_stats - takes one argument - a raw disk device;
#                   prints the size (in kilobytes) and type of the disk
#
get_disk_stats()
{

  if [ "$uid" -ne 0 ]; then
     return 1
  fi
  echo `diskinfo $rdsk 2>/dev/null | sed 's/^[ ]*//' | sort | \
     awk '{if(match($1,"(size|type):")){print $2,$3,$4,$5}}'` | \
	  read disksize junk disktype
  if [ -z "$disksize" ] || [ -z "$disktype" ]; then
     return 2
  fi
  echo "$disksize $disktype"
  return 0

}  # end get_disk_stats()


print_patches()
{

   echo "$divider\c"

   if [ "$OSNAME" = SunOS ]; then
      echo "Patches:"
      showrev -p
      return 0
   fi

   if [ "$OSVER" -ge 1111 ]; then
      xswhwcr_bundle=HWEnable11i
      xswgr_bundle=GOLDBASE11i
   elif [ "$OSVER" -ge 1100 ]; then
      xswhwcr_bundle="(XSWHWCR$OSVER|HWE$OSVER)"
      xswgr_bundle=XSWGR$OSVER
   else
      if [ "$SERIES" = 800 ]; then
         xswhwcr_bundle=XSW800HWCR$OSVER
      else
         xswhwcr_bundle=XSW700HW$OSVER
      fi
      xswgr_bundle=XSW${SERIES}GR$OSVER
   fi
   if [ "$OSVER" -ge 1111 ]; then
      qpk_bundle=GOLDAPPS11i
   elif [ "$OSVER" -ge 1100 ]; then
      qpk_bundle=QPK$OSVER
   else
      qpk_bundle=${SERIES}QPK$OSVER
   fi
   swlist_output=`mktemp`
   if grep show_superseded_patches /usr/lib/sw/sys.defaults \
      >/dev/null 2>&1; then
      swlist_opts="-x show_superseded_patches=false"
   fi
   if swlist $swlist_opts >$swlist_output 2>/dev/null; then
      echo "Patch bundles:"
      grep -E " $xswhwcr_bundle" $swlist_output
      grep -E " $xswgr_bundle" $swlist_output
      grep -E " $qpk_bundle" $swlist_output
      echo "Other patches:"
      grep " PH.._" $swlist_output
      _rc=0
   else
      echo "  <access_failed>"
      _rc=1
   fi
   rm -f $swlist_output
   return $_rc

}  # end print_patches()


print_disk_info()
{

   [ "$OSNAME" = SunOS ] && return 1

   echo "$divider\c"

   fixed_disks=0
   total_disksize=0
   dsks=`ioscan -fnkC disk 2>/dev/null | grep /dev/dsk/ | \
    awk '{print $1}'`
   let x=0
   bootpath1=`setboot 2>/dev/null | grep Primary | awk '{print $4}'`
   found1=0
   bootpath2=`setboot 2>/dev/null | grep Alternate | awk '{print $4}'`
   found2=0
   altlinks=`vgdisplay -v 2>/dev/null | grep "Alternate Link" | \
    awk '{print $3}'`
   found3=0

for dsk in $dsks; do
   rdsk=`echo $dsk | sed 's:/dsk/:/rdsk/:'`
   ioscan -k $rdsk 2>/dev/null | grep disk | read hwpath junk
   boot=
   if [ "$hwpath" = "$bootpath1" ]; then
      boot="(*)"
      found1=1
   elif [ "$hwpath" = "$bootpath2" ]; then
      boot="(@)"
      found2=1
   fi
   u=
   if [ "$x" -lt 10 ]; then
      u=_
   fi
   echo "disk_$u$x   : \c"
   [ "$show_status" = 1 ] && \
    echo "<working...>\b\b\b\b\b\b\b\b\b\b\b\b\c"
   get_disk_stats $rdsk | read disksize disktype
   rc=$?
   alt=
   if echo $altlinks | grep -q $dsk; then
      alt="(+)"
      found3=1
   fi
   if [ "$rc" -eq 0 ]; then
      # note: it is important to add the disk size to the running total
      #       _before_ converting to larger units
      if [ -z "$alt" ] && [ "$disktype" != CD-ROM ]; then
         total_disksize=`expr $total_disksize + $disksize`
         let fixed_disks=fixed_disks+1
      fi
      if [ "$disksize" -eq 0 ]; then
         disksize="0 (no disk is mounted)"
      else
         disksize=`convert_kilobytes $disksize`
      fi
   else
      #  see why get_disk_stats() failed
      if [ "$rc" -eq 1 ]; then
         failure_reason="<access_denied>"
      elif [ "$rc" -eq 2 ]; then
         failure_reason="<access_failed>"
      fi
      disksize=$failure_reason
      disktype=$failure_reason
   fi
   echo "$rdsk; $disksize; $disktype\c"
   [ -n "$boot" ] && echo " $boot\c"
   [ -n "$alt" ] && echo " $alt\c"
   echo
   let x=x+1
done

echo
[ "$found1" -eq 1 ] && echo "(*) = primary boot device"
[ "$found2" -eq 1 ] && echo "(@) = secondary boot device"
[ "$found3" -eq 1 ] && echo "(+) = redundant device"

if [ "$total_disksize" -eq 0 ]; then
   total_disksize="<?>"
   fixed_disks="<?>"
else
   total_disksize=`convert_kilobytes $total_disksize`
fi
echo
echo "Total fixed disks      : $fixed_disks"
echo "Total fixed disk space : $total_disksize"

}  # end print_disk_info()


get_pa_stats()
{

MODEL=`uname -m | cut -d/ -f2`

LINE=`grep "^$MODEL" <<EoF
/* /samsrc/mo/data/sched.models 73.18 2001-01-30 16:11:28-07 */
600     1.0     PA7000
635     1.0     PA7000
645     1.0     PA7000
700     1.1     PA7000
705     1.1a    PA7000
715     1.1c    PA7100LC
710     1.1a    PA7000
712     1.1c    PA7100LC
720     1.1a    PA7000
722     1.1c    PA7100LC
725     1.1c    PA7100LC
728     1.1d    PA7200
730     1.1a    PA7000
735     1.1b    PA7100
742     1.1b    PA7100
743     1.1c    PA7100LC
744     1.1e    PA7300
745     1.1b    PA7100
747     1.1b    PA7100
750     1.1a    PA7000
755     1.1b    PA7100
770     1.1d    PA7200
777     1.1d    PA7200
778     1.1e    PA7300
779     1.1e    PA7300
780     2.0     PA8000
781     2.0     PA8000
782     2.0     PA8200
800     1.0     PA7000
801     1.1c    PA7100LC
802     2.0     PA8000
803     1.1e    PA7300
804     2.0     PA8000
806     1.1c    PA7100LC
807     1.1a    PA7000
808     1.0     PA7000
809     1.1d    PA7200
810     2.0     PA8000
811     1.1c    PA7100LC
813     1.1e    PA7300
815     1.0     PA7000
816     1.1c    PA7100LC
817     1.1a    PA7000
819     1.1d    PA7200
820     2.0     PA8000
821     1.1d    PA7200
822     1.0     PA7000
825     1.0     PA7000
826     1.1c    PA7100LC
827     1.1a    PA7000
829     1.1d    PA7200
831     1.1d    PA7200
832     1.0     PA7000
834     1.0     PA7000
835     1.0     PA7000
837     1.1a    PA7000
839     1.1d    PA7200
840     1.0     PA7000
841     1.1d    PA7200
842     1.0     PA7000
845     1.0     PA7000
847     1.1a    PA7000
849     1.1d    PA7200
850     1.0     PA7000
851     1.1d    PA7200
852     1.0     PA7000
855     1.0     PA7000
856     1.1c    PA7100LC
857     1.1a    PA7000
859     1.1d    PA7200
860     1.0     PA7000
861     2.0     PA8000
865     1.0     PA7000
867     1.1a    PA7000
869     1.1d    PA7200
870     1.0     PA7000
871     2.0     PA8000
877     1.1a    PA7000
879     2.0     PA8000
887     1.1b    PA7100
889     2.0     PA8000
890     1.0     PA7000
891     1.1b    PA7100
892     1.1b    PA7100
893     2.0     PA8000
897     1.1b    PA7100
898     2.0     PA8200
899     2.0     PA8200
F10     1.1a    PA7000
F20     1.1a    PA7000
H20     1.1a    PA7000
F30     1.1a    PA7000
G30     1.1a    PA7000
H30     1.1a    PA7000
I30     1.1a    PA7000
G40     1.1a    PA7000
H40     1.1a    PA7000
I40     1.1a    PA7000
G50     1.1b    PA7100
H50     1.1b    PA7100
I50     1.1b    PA7100
G60     1.1b    PA7100
H60     1.1b    PA7100
I60     1.1b    PA7100
G70     1.1b    PA7100
H70     1.1b    PA7100
I70     1.1b    PA7100
E25     1.1c    PA7100LC
E35     1.1c    PA7100LC
E45     1.1c    PA7100LC
E55     1.1c    PA7100LC
T500    1.1b    PA7100
T520    1.1b    PA7100
T540    2.0     PA8000
K100    1.1d    PA7200
K200    1.1d    PA7200
K210    1.1d    PA7200
K220    1.1d    PA7200
K230    1.1d    PA7200
K400    1.1d    PA7200
K410    1.1d    PA7200
K420    1.1d    PA7200
DXO     1.1c    PA7100LC
DX0     1.1c    PA7100LC
DX5     1.1c    PA7100LC
D200    1.1c    PA7100LC
D210    1.1c    PA7100LC
D310    1.1c    PA7100LC
D410    1.1d    PA7200
D250    1.1d    PA7200
D350    1.1d    PA7200
J200    1.1d    PA7200
J210    1.1d    PA7200
C100    1.1d    PA7200
J220    2.0     PA8000
J280    2.0     PA8000
S715    1.1e    PA7300
S760    1.1e    PA7300
D650    2.0     PA8000
J410    2.0     PA8000
J400    2.0     PA8000
J210XC  1.1d    PA7200
J2240   2.0     PA8200
C200+   2.0     PA8200
C240+   2.0     PA8200
C180    2.0     PA8000
C180-XP 2.0     PA8000
C160    2.0     PA8000
C160L   1.1e    PA7300
C140    2.0     PA8000
C130    2.0     PA8000
C120    1.1e    PA7300
C115    1.1e    PA7300
C110    1.1d    PA7200
C360    2.0     PA8500
B160L   1.1e    PA7300
B132L   1.1e    PA7300
B120    1.1e    PA7300
B115    1.1e    PA7300
S700i   1.1e    PA7300
S744    1.1e    PA7300
D330    1.1e    PA7300
D230    1.1e    PA7300
D320    1.1e    PA7300
D220    1.1e    PA7300
D360    1.1d    PA7200
K360    2.0     PA8000
K370    2.0     PA8200
K460    2.0     PA8000
K460-EG 2.0     PA8000
K460-XP 2.0     PA8000
K260    2.0     PA8000
K260-EG 2.0     PA8000
D260    1.1d    PA7200
D270    2.0     PA8000
D280    2.0     PA8000
D370    2.0     PA8000
D380    2.0     PA8000
D390    2.0     PA8000
R380    2.0     PA8000
R390    2.0     PA8000
K250    2.0     PA8000
K450    2.0     PA8000
K270    2.0     PA8200
K470    2.0     PA8200
K380    2.0     PA8200
K580    2.0     PA8200
V2200   2.0     PA8200
V2250   2.0     PA8200
V2500   2.0     PA8500
V2600   2.0     PA8600
V2650   2.0     PA8700
V2700   2.0     PA8700
A180    1.1e    PA7300LC
A180c   1.1e    PA7300LC
A400-36 2.0     PA8500
A400-44 2.0     PA8500
A400-5X 2.0     PA8600
A400-7X 2.0     PA8700
A400-8X 2.0     PA8700
A500-36 2.0     PA8500
A500-44 2.0     PA8500
A500-55 2.0     PA8600
A500-5X 2.0     PA8600
A500-7X 2.0     PA8700
A500-8X 2.0     PA8700
B1000   2.0     PA8500
B2000   2.0     PA8500
B2600   2.0     PA8700
C3000   2.0     PA8500
C3600   2.0     PA8600
C3700   2.0     PA8700
C3750   2.0     PA8700
J5000   2.0     PA8500
J5600   2.0     PA8600
J6000   2.0     PA8600
J6700   2.0     PA8700
J6750   2.0     PA8700
J7000   2.0     PA8500
J7600   2.0     PA8600
SD16000 2.0     PA8600
SD32000 2.0     PA8600
SD64000 2.0     PA8600
S16K-A  2.0     PA8700
N8K-A   2.0     PA8700
N4000-36        2.0     PA8500
N4000-44        2.0     PA8500
N4000-55        2.0     PA8600
N4000-5X        2.0     PA8600
N4000-7X        2.0     PA8700
N4000-8X        2.0     PA8700
N4000-8Y        2.0     PA8700
N4000-8Z        2.0     PA8700
N4000-9X        2.0     PA8700
L1000-36        2.0     PA8500
L1000-44        2.0     PA8500
L1000-5X        2.0     PA8600
L1000-8X        2.0     PA8700
L1500-7x        2.0     PA8700
L1500-8x        2.0     PA8700
L1500-9x        2.0     PA8700
L2000-36        2.0     PA8500
L2000-44        2.0     PA8500
L2000-5X        2.0     PA8600
L2000-8X        2.0     PA8700
L3000-55        2.0     PA8600
L3000-5x        2.0     PA8600
L3000-7x        2.0     PA8700
L3000-8x        2.0     PA8700
L3000-9x        2.0     PA8700
EoF
`

echo $LINE | cut -f2,3 -d' ' 2>/dev/null
return 0

}  # end get_pa_stats()


get_jvm()
{

   if [ "$OSNAME" = HP-UX ]; then
      javadirs="/opt/java1.4 /opt/java1.3 /opt/java1.2 /opt/java"
   elif [ "$OSNAME" = SunOS ]; then
      javadirs="/usr/j2se /usr/java1.2 /usr/java /usr/j2sdk1.4.0 /usr/j2sdk1.3.1 /usr/j2sdk1.3.0 /opt/j2sdk1.4.0 /opt/j2sdk1.3.1 /opt/j2sdk1.3.0"
   fi
   JAVA_HOME=
   for dir in $javadirs; do
      if [ -f $dir/bin/java ]; then
         JAVA_HOME=$dir
         break
      fi
   done
   if [ -z "$JAVA_HOME" ]; then
      PATH=$SAVED_PATH
      JAVA=$(whence java)
      PATH=$SYSINFO_PATH
   else
      JAVA=${JAVA_HOME}/bin/java
   fi
   [ -x "$JAVA" ] && JAVA_VER=`$JAVA -version 2>&1 | head -1 | \
    cut -d'"' -f2`
   if [ -n "$JAVA_VER" ]; then
      if [ "`echo $JAVA_VER | cut -d'-' -f1`" = JavaVM ]; then
         JAVA_VER=`echo $JAVA_VER | cut -d'-' -f2`
      elif [ "`echo $JAVA_VER | cut -d' ' -f1-2`" = "HP-UX Java" ]; then
         JAVA_VER=`echo $JAVA_VER | cut -d' ' -f3 | \
		  sed 's/[A-Z]\.01\.1/1.1./'`
      fi
      echo "Java $JAVA_VER ($JAVA)"
      return 0
   else
      echo "<none_found>"
      return 1
   fi

}  # end get_jvm()


build_mem()
{

   BINDIR="/var/tmp/`whoami`/bin"
   mkdir -p $BINDIR
   MEM=$BINDIR/mem
   cat <<-EoF >$MEM.c
	#include <errno.h>
	#include <stdio.h>
	#include <sys/param.h>
	#include <sys/pstat.h>
	#define BYTES_PER_MB 1048576
	main()
	{
		struct pst_static pst;
		union pstun pu;
		pu.pst_static = &pst;
		if ( pstat( PSTAT_STATIC, pu, (size_t)sizeof(pst), (size_t)0, 0 ) != -1 ) {
			printf( "%ld MB\n", 
			(long)( (double)pst.physical_memory * pst.page_size / BYTES_PER_MB ) );
			exit( 0 );
		} else {
			perror("pstat_getstatic");
			exit( errno );
		}
	}
	EoF
   cc -o $MEM $MEM.c >/dev/null 2>&1
   rm $MEM.c

}  # end build_mem()


get_general_stats()
{

echo "$divider\c"

#  get OS name, release number, & kernel bits
OS=`uname -sr`

#  determine if the kernel is 32 or 64 bit
if [ "$OSNAME" = HP-UX ]; then
   if [ "$OSVER" -ge 1100 ]; then
      KERNEL_BITS=`getconf KERNEL_BITS`
      OS="$OS (${KERNEL_BITS}-bit)"
   else
      KERNEL_BITS=32
   fi
elif [ "$OSNAME" = SunOS ]; then
   KERNEL_BITS=`isainfo -b`
   OS="$OS (${KERNEL_BITS}-bit)"
fi

#  add VirualVault version info if OS is VVOS
if [ "$OSNAME" = HP-UX ] && [ "$OSVER" = 1024 ] || \
 [ "$OSVER" = 1104 ]; then
   VV_VER=
   if [ "$OSVER" = 1024 ]; then
      VV_BUNDLES="VaultSAFE-US VaultSAFE-INTL"
      VV_BUNDLE_DESC=VirtualVault/SAFE
   else
      VV_BUNDLES="B5411DA B5412DA"
      VV_BUNDLE_DESC="VirtualVault Master Bundle"
   fi
   VV_VER=`swlist $VV_BUNDLES 2>/dev/null | grep "$VV_BUNDLE_DESC" | \
      awk '{print $3}'`
   [ -n "$VV_VER" ] && VV="VirtualVault $VV_VER"
   OS="$OS ($VV)"
fi

#  see if a JVM is installed
JVM=`get_jvm`

#  if possible, get system stats directly from kernel
get_kernel_stats
rc=$?

#  if not, try to get as much info as we can by other means
if [ "$rc" -eq 0 ]; then
   return 0
elif [ "$rc" -eq 1 ]; then
   failure_reason="<access_denied>"
elif [ $"rc" -eq 2 ]; then
   failure_reason="<access_failed>"
fi
hostname=`hostname`
if [ "$OSNAME" = HP-UX ]; then
      if [ -x /usr/bin/model ]; then
         model -D | read model model_desc
         model_desc=`echo $model_desc | sed 's/(no description)//`
         if [ -n "$model_desc" ]; then
            model="$model $model_desc"
         else
            model="$model $desc"
         fi
      else
         model="`uname -m` $desc"
      fi
      get_pa_stats | read CPU_arch CPU_model
      CPU_speed=$failure_reason
      CPU_count=`ioscan -kC processor | grep processor | wc -l`
      CPU="($CPU_count) $CPU_model $CPU_speed MHz (PA-RISC $CPU_arch)"
      build_mem
      RAM=`$MEM`
      rm $MEM
      [ $? -ne 0 ] && RAM="<access_failed>"
elif [ "$OSNAME" = SunOS ]; then
   # use read to remove leading whitespace in wc output
   psrinfo | wc -l | read CPU_count
   psrinfo -v | head -4 | grep operates | \
      read j1 CPU_arch j3 j4 j5 CPU_speed j7
   CPU_arch=`echo $CPU_arch | sed 's/sparcv//'`
   class=`uname -m`
   PRTDIAG=/usr/platform/${class}/sbin/prtdiag
   if [ -f "$PRTDIAG" ]; then
      CPU_model=`$PRTDIAG | head -1 | cut -d '(' -f2 | awk '{print $3}'`
      $PRTDIAG | head -1 | sed "s/.*${class}//" | cut -d'(' -f1 | \
	   read model 
   fi
   CPU="($CPU_count) $CPU_model $CPU_speed MHz (SPARC V$CPU_arch)"
   prtconf | grep ^Memory | read j1 j2 RAM units
   if [ "$units" = Megabytes ]; then
      RAM=`convert_megabytes $RAM`
   fi
fi

}  # end get_general_stats()


print_general_stats()
{

   get_general_stats
   echo "hostname : $hostname"
   echo "model    : $model"
   echo "CPU      : $CPU"
   echo "RAM      : $RAM"
   echo "OS       : $OS"
   echo "JVM      : $JVM"

}  # end print_general_stats()


print_version()
{

   echo "$divider\c"
   echo "sysinfo version $sysinfo_version"

}  # end print_version()


print_help()
{

   echo "$divider\c"
   sysinfo=`basename $0`
   echo "usage: $sysinfo [-gdpaqv?]"
   echo
   echo "  -g : general stats (default)"
   echo "  -d : disk info"
   echo "  -n : network info (IP, MAC)"
   echo "  -p : patches installed"
   echo "  -a : all of the above"
   echo "  -q : quiet mode (no progress status; use if redirecting" \
    "output to a file)"
   echo "  -v : sysinfo version"
   echo "  -? : describe options"

}  # end print_help()


print_network_info()
{
   integer Ncount=0

   echo "$divider\c"

   if [ "OSNAME" = "SunOS" ]; then
      echo "IP address: Not Implemented yet"
      echo "MAC address: Not Implemented yet"
      return
   fi   

   lanscan | egrep -v "^Hardware|^Path" |
     while read Npath Nmac Ncrd Nstate Ndev Nid Ntype Njunk; do

        # check that the IP device isn't NULL
        if [ "${Ndev}" != "" ]; then
           ifconfig ${Ndev} 2>&1 | tail -1 | read Njunk1 Nip Njunk2
        fi

        # check if Nip is null or if 'ifconfig' reported an error
        if [ "${Nip}" = "" -o \
             "${Njunk1}" = "ifconfig:" -o \
             "${Njunk1}" = "usage:" ]; then

               Nip="<unavailable>"
               Nname="<unavailable>"
        else
               nslookup ${Nip} 2>&1 | grep "^Name:" | read Njunk1 Nname
               if [ "${Nname}" = "" ]; then
                  Nname="<unavailable>"
               else
                  Nname="$Nname"
               fi
        fi

        if [ "${Nmac}" = "" ]; then
           Nmac="<unavailable>"
        else
           N1=`echo $Nmac | sed -e \
             "s/0x\\(..\\)\\(..\\)\\(..\\)\\(..\\)\\(..\\)\\(..\\)/\\1-\\2-\\3-\\4-\\5-\\6/" `
        fi

        echo "IP_addr[$Ncount]  : $Nip (${Nname})"
        echo "MAC_addr[$Ncount] : ${N1}"

        let Ncount+=1

   done

} # end print_network_info()


usage()
{

   this=`basename $0`
   echo "usage $this [-gdpaqv?]" >&2
   exit 1

}  # end usage()


#################################{ MAIN }################################

SYSINFO_PATH=/usr/bin:/usr/sbin
SAVED_PATH=$PATH
PATH=$SYSINFO_PATH

#  check if this is a supported OS (currently HP-UX 10+ or Solaris 7+)
OSNAME=`uname -s`
OSVER=`uname -r | sed 's/[A-Z.]//g'`

if [ "$OSNAME" = HP-UX ]; then 
   case $OSVER in
      09??)                       # HP-UX 9.x
         KERNEL=/hp-ux
      ;;
      10??|11??)                  # HP-UX 10.x or 11.x
         KERNEL=/stand/vmunix   
      ;;
      *)                          # HP-UX < 9.0
         echo "This script only supports HP-UX version 9.0 or later." >&2
         exit 1
      ;;
   esac
   SERIES=`uname -m | cut -d/ -f2 | cut -c1`00
   if [ "$SERIES" = 700 ]; then
      desc="(workstation)"
   elif [ "$SERIES" = 800 ]; then
      desc="(server)" 
   fi
elif [ "$OSNAME" = SunOS ]; then
   case $OSVER in
      5[789])                     # Solaris 7+
         KERNEL=/dev/ksyms
      ;;
      *)                          # Solaris < 7
         echo "This script only supports Solaris version 7 or later." >&2
         exit 1
      ;;
   esac
else
   echo "This script only runs on HP-UX 10+ and Solaris 7+." >&2
   exit 1
fi

#  get current user's UID
uid=`id | cut -d= -f2 | cut -d'(' -f1`

show_version=0
show_help=0
show_general_stats=0
show_disk_info=0
show_patches=0
if [ $# -eq 0 ]; then
   show_general_stats=1
else
   show_general_stats=0
fi
show_status=1

while getopts :gd\panqv? opt; do
   case "$opt" in
      g) show_general_stats=1 ;;
      d) show_disk_info=1 ;;
      n) show_network_info=1 ;;
      p) show_patches=1 ;;
      a) show_general_stats=1
         show_disk_info=1
         show_network_info=1
         show_patches=1
         ;;
      q) show_status=0 ;;
      v) show_version=1 ;;
      \?) show_help=1 ;;
      *) usage
   esac
done
shift $(($OPTIND -1))

hr=
typeset -i x=0
eval `stty -a | grep columns | cut -d';' -f2 | sed 's/ //g'`
while [ "$x" -lt "$columns" ]; do
  hr="${hr}-"
  let x=x+1
done
divider=
  
if [ "$show_version" = 1 ]; then
   print_version
   divider="$hr"
fi
if [ "$show_help" = 1 ]; then
   print_help
   divider="$hr"
fi
if [ "$show_general_stats" = 1 ]; then
   print_general_stats
   divider="$hr"
fi
if [ "$show_network_info" = 1 ]; then
   print_network_info
   divider="$hr"
fi
if [ "$show_disk_info" = 1 ]; then
   print_disk_info
   divider="$hr"
fi
if [ "$show_patches" = 1 ]; then
   print_patches
   divider="$hr"
fi

#  exit w/ success
exit 0
